/*************************************************************************
 * The contents of this file are subject to the MYRICOM MYRINET          *
 * EXPRESS (MX) NETWORKING SOFTWARE AND DOCUMENTATION LICENSE (the       *
 * "License"); User may not use this file except in compliance with the  *
 * License.  The full text of the License can found in LICENSE.TXT       *
 *                                                                       *
 * Software distributed under the License is distributed on an "AS IS"   *
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied.  See  *
 * the License for the specific language governing rights and            *
 * limitations under the License.                                        *
 *                                                                       *
 * Copyright 2003 - 2004 by Myricom, Inc.  All rights reserved.          *
 *************************************************************************/

#ifndef _mx_instance_h_
#define _mx_instance_h_

#include <mx_pci.h>
#include <mx_mcp_interface.h>

#include "mx_pin.h"

#ifndef MX_DISABLE_ETHERNET
#define MX_DISABLE_ETHERNET 0
#endif
#ifndef MX_HAS_PCIE_LINK_RESET
#define MX_HAS_PCIE_LINK_RESET 0
#endif

/* forward declaration */
typedef struct mx_instance_state mx_instance_state_t;
typedef struct mx_endpt_state mx_endpt_state_t;
typedef struct mx_waitq mx_waitq_t;
typedef struct mx_page_pin mx_page_pin_t;
typedef struct mx_mcp_dma_chunk mx_mcp_dma_chunk_t;
typedef struct mx_host_dma_win mx_host_dma_win_t;
struct mx_ether;

typedef struct mx_intrq
{
  mcp_slot_t *q[2];
  int intrq;
  int slot;
  int maxslots;
  uint32_t seqnum;
  uint32_t spurious;
  uint32_t cnt;
  mx_page_pin_t pin;
  char *alloc_addr;
} mx_intrq_t;

typedef struct mx_copyblock
{
  void *alloc_addr;
  void *addr;
  uint32_t size;
  mx_page_pin_t *pins;
} mx_copyblock_t;

typedef struct mx_routes
{
  char *host_table;
  char *mcp_table;
  int *indexes;
  int *offsets;
  int block_size;
  mx_sync_t cmd_sync;
} mx_routes_t;

typedef struct mx_raw_rx
{
  int length;
  int port;
} mx_raw_rx_t;

typedef int (mx_init_board_t)(mx_instance_state_t *is);
typedef int (mx_map_board_t)(mx_instance_state_t *is);
typedef void (mx_unmap_board_t)(mx_instance_state_t *is);
typedef void (mx_claim_interrupt_t)(mx_instance_state_t *is);
typedef int (mx_might_be_interrupting_t)(mx_instance_state_t *is);
typedef void (mx_enable_interrupt_t)(mx_instance_state_t *is);
typedef void (mx_disable_interrupt_t)(mx_instance_state_t *is);
typedef void (mx_park_board_t)(mx_instance_state_t *is);
typedef int (mx_detect_parity_t)(mx_instance_state_t *is);
typedef void (mx_dump_registers_t)(mx_instance_state_t *is, 
				   uint32_t *reg, uint32_t *cnt);
typedef void (mx_get_freq_t)(mx_instance_state_t *is);
typedef void (mx_write_kreq_t)(mx_instance_state_t *is, mcp_kreq_t *kreq);

typedef struct mx_board_ops {
  mx_init_board_t 		*init;
  mx_map_board_t 		*map;
  mx_unmap_board_t 		*unmap;
  mx_claim_interrupt_t 		*claim_interrupt;
  mx_might_be_interrupting_t 	*might_be_interrupting;
  mx_enable_interrupt_t 	*enable_interrupt;
  mx_disable_interrupt_t 	*disable_interrupt;
  mx_park_board_t		*park;
  mx_detect_parity_t		*detect_parity_error;
  mx_dump_registers_t		*dump_registers;
  mx_get_freq_t		        *get_freq;
  mx_write_kreq_t		*write_kreq;
} mx_board_ops_t;

extern mx_board_ops_t mx_lx_ops;
extern mx_board_ops_t mx_lz_ops;
void mx_lx_init_board_ops(void);
void mx_lz_init_board_ops(void);

#define MX_ADDRS_PER_VPAGE  (MX_VPAGE_SIZE / sizeof(mcp_dma_addr_t))
#define MX_ADDRS_PER_VPAGE_SHIFT  (MX_VPAGE_SHIFT - MX_MCP_DMA_ADDR_SHIFT)

#define MX_NUM_DIRECT_DMA  4  /* XXXX move me elsewhere!!!! */
#define MX_MAX_MAPPER_TYPES 5

/* An mx_dma_table contains one page of DMA addresses.
   Lengths for each DMA are implicitly assumed to be VPAGE size.
   The MCP accesses this structure, so it must be contiguous in DMA space.  
   The table can be a pte table or a higher-level table */

struct mx_dma_table
{
  char *alloc_addr;		/* the address of the DMA page */
  mx_page_pin_t pin;
  mcp_dma_addr_t *desc;		/* pointer to DMA page */
  union {
    mx_page_pin_t *pins; /* DMA pin info for data pages */
    struct mx_dma_table **tables; /* tables of inferor lvel  */
  } u;
  int nb_entries;
  int nb_subtables;
  int log_level; /* 0=> table of data pages , n*MX_ADDRS_PER_VPAGE_SHIFT => n-higher-level table */
};

typedef struct mx_dma_table mx_dma_table_t;

/* An mx_host_dma_win_t is a container for one rdma window,
   it contains the page table associated with this window. */

struct mx_host_dma_win
{
  uint32_t flags;
  struct mx_dma_table table;
};

/* Queued lanai commands that are buffered in the host.
   We use a tailq because linux does not have slists
   in sys/queue.h */

struct cmdq_entry;
typedef void (mx_cmd_func_t)(void *arg, struct cmdq_entry *);

struct cmdq_entry
{
  STAILQ_ENTRY(cmdq_entry) entries;
  mx_sync_t *sync;
  mx_cmd_func_t *func;		   
  uint32_t mcp_status;
  uint32_t mcp_data;
  uint32_t slot;
  uint32_t cmd;
  uint32_t cmd_data0; 
  uint32_t cmd_data1;
  uint32_t cmd_seqnum;
  uint32_t cmd_index;
};

STAILQ_HEAD(cmdq_head, cmdq_entry); 

struct raw_eventq_entry;

struct raw_eventq_entry
{
  STAILQ_ENTRY(raw_eventq_entry) entries;
  mx_raw_next_event_t event;
};
STAILQ_HEAD(raw_eventq_head, raw_eventq_entry);

typedef struct mx_rawbuf {
  mcp_dma_addr_t dma;
  void *buf;
} mx_rawbuf_t;


struct mx_raw_vpage {
    char *alloc_addr;
    char *buf;
    mx_page_pin_t pin;
};

struct mx_raw_info {
  mx_endpt_state_t *es;
  struct mx_raw_vpage *vpages;
  int num_vpages;
  struct raw_eventq_head free_events;
  struct raw_eventq_head pending_events;
  mx_spinlock_t spinlock;
  mx_rawbuf_t *rx_bufs;
  mx_rawbuf_t *tx_bufs;
  int rx_cnt;
  int max_rx_idx;
  int host_rx;
  int tx_req;
  int tx_done;
  int max_tx_idx;
  int wakeup_needed;
  mx_mapper_state_t mapper_state[2];
  int valid_route_count[2];
  int use_count;
  mx_sync_t sync;
};

#define MX_IS_DEAD        (1 << 0)
#define MX_IS_LOGGING     (1 << 1)
#define MX_HOSTNAME_SET   (1 << 2)
#define MX_PARITY_RECOVERY (1 << 3)

#define MX_INTR_NONE 0 
#define MX_INTR_MAYBE 1
#define MX_INTR_ACTIVE 2

struct mx_instance_state
{
  unsigned int id;            /* unit number.         */
  unsigned int flags;         /* Flags, defined above */
  mx_board_ops_t board_ops;

  /*************
   * LANai access fields
   *************/

  /* Record of lanai mappings */
    
  struct
  {
    /* This pointer is opaque to ensure that the special registers
       are referenced only through the pio.h interface. */
    void *special_regs;
    volatile uint32_t *control_regs;
    volatile uint8_t *sram;
    /* The globals are not a structure to insure that they are
       referenced only throught the pio.h interface. */
    volatile void *globals; /* Set iff MCP running. */
    uint32_t running;
    char *eeprom_strings;
    char *product_code;
    char *part_number;
    uint32_t serial;
  }
  lanai;

  /* Pointer to LANai interface card, mapped into kernel memory. */

  uint32_t board_span;
  uint32_t sram_size;
  uint32_t specials_size;
  uint32_t using_msi;
  uint32_t msi_enabled;
  uint32_t cpu_freq;
  uint32_t pci_freq;
  mx_arch_instance_info_t arch; /* Architecture-specific field. */
  mx_sync_t sync;
  mx_endpt_state_t **es;  
  uint8_t mac_addr[8];
  char *mac_addr_string;
  mx_intrq_t intr;
  struct {
    mcp_slot_t *mcp;
    mx_spinlock_t spinlock;
    int submitted; /* refers to just mcp_cmdq; */
    int completed; /* refers to just mcp_cmdq; */
    int max_index;
    int len;	   /* refers to both mcp_cmdq and host_cmdq */
    struct cmdq_head host_cmdq;
    struct cmdq_head mcp_cmdq;
  } cmdq;
  mcp_kreq_t *kreqq;
  int kreqq_submitted;
  int kreqq_completed;
  int kreqq_max_index;
  mx_spinlock_t kreqq_spinlock;
  int cmdq_inflight;
  struct {
    mx_sync_t sync;
    uint32_t size;
  } logging;
  mx_sync_t init_sync;
  mx_atomic_t ref_count;
  struct mx_raw_info raw;
  struct {
    char *alloc_addr;
    char *buf;
    mx_page_pin_t pin;
  } host_query;
  int query_pending;
  uint32_t num_ports;
  uint8_t pci_rev;
  uint16_t pci_devid;
  uint32_t bar0_low;
  uint32_t bar0_high;
  uint32_t bar2_low;
  uint32_t bar2_high;
  uint32_t msi_addr_low;
  uint32_t msi_addr_high;
  uint16_t msi_data;
  uint16_t msi_ctrl;
  uint32_t ze_print_pos;
  int board_type;
  mx_routes_t *routes;
  uint32_t mcp_print_len;
  uint32_t mcp_print_addr;
  uint32_t mcp_print_idx;
  uint32_t mcp_print_limit_addr;
  char *mcp_print_buffer;
  volatile uint32_t *lanai_uptime_ptr;
  uint32_t old_lanai_uptime;
  struct mx_ether *ether;
  int ether_is_open;
  struct {
    mx_sync_t wait_sync;
    mx_sync_t cmd_sync;
    uint32_t cycles;
    uint32_t count;
    int busy;
  } dmabench;
  mcp_dma_addr_t *rdmawin_vpages_dma;
  struct mx_rdmawin_vpage {
    mx_page_pin_t pin;
    mcp_rdma_win_t *rdmawins;
    char *alloc_addr;
  } *rdmawin_vpages;
  int rdmawin_vpages_count;
  int parity_errors_detected;
  int parity_errors_corrected;
  uintptr_t parity_critical_start;
  uintptr_t parity_critical_end;
  struct {
    int command_tweaked;
    uint16_t command_offset;
    uint16_t command_orig_val;
  } pci_cap;
  uint32_t link_state;
  struct {
    int reason;
    int arg;
    uint32_t num_registers;
    uint32_t *registers;
  } saved_state;
  struct  {
    mx_page_pin_t pin;
    char *alloc;
    char *addr;
  } bogus;
  mx_kernel_window_t *kernel_window;
};

#define MX_RDMA_WIN_PER_PAGE (MX_VPAGE_SIZE/sizeof(mcp_rdma_win_t))

#define MX_CTL 254
#define MX_CTLP 255

#define MX_ES_INVALID	0
#define MX_ES_API	(1 << 0)
#define MX_ES_RAW	(1 << 1)
#define MX_ES_CLOSING	(1 << 2)

struct mx_endpt_state
{
  mx_instance_state_t *is;
  uint32_t endpt;
  uint32_t flags;
  uint32_t ref_count;  /* access protected by sync */
  mx_sync_t sync;

  /* copyblocks */
  mx_copyblock_t sendq;
  mx_copyblock_t recvq;
  mx_copyblock_t eventq;
  mx_copyblock_t user_mmapped_sram;  /* sram, not kernel mem */
  mx_copyblock_t user_mmapped_zreq;  /* mmio, not kernel mem */


  /* waiting */
  mx_sync_t cmd_sync;
  mx_sync_t wait_sync;
  mx_sync_t app_wait_sync;
  mx_atomic_t no_mcp_req_wakeups;
  unsigned progression_timeout;
  int app_waiting;
  mx_sync_t close_sync;
  uint32_t num_waiters;

  /* MD handle */
  mx_arch_endpt_info_t arch;

  /* Memory locked by large send/recv indexed by handle*/
  struct {
    mx_host_dma_win_t *win;
    uint16_t seqno;
  } *host_rdma_windows;

  /* where to write the vpages containing user-area dma mappings */
  mcp_rdma_win_t *mcp_rdma_windows;

  uint32_t privileged;
  uint32_t is_kernel;
  mx_opener_t opener;
  int parity_errors_detected;
  int parity_errors_corrected;
  int endpt_error;
  uint32_t session_id;
};

int mx_init_driver(void);
int mx_finalize_driver(void);

int  mx_instance_init (mx_instance_state_t *is, int32_t unit);
int  mx_instance_finalize (mx_instance_state_t *is);
void mx_lanai_print(mx_instance_state_t *is, int idx);
int mx_load_mcp(mx_instance_state_t *is, void * img, int limit, int *actual_len);


void *mx_map_pci_space (mx_instance_state_t * is, int bar, uint32_t offset, 
		       uint32_t len);

void *mx_map_io_space (mx_instance_state_t * is, uint32_t offset, 
		       uint32_t len);
void  mx_unmap_io_space (mx_instance_state_t * is,
			 uint32_t len, void *kaddr);

extern mx_instance_state_t **mx_instances;
extern uint32_t mx_max_instance;
extern uint32_t mx_num_instances;
extern uint32_t mx_mapper_mask;
extern uint32_t mx_mapper_running_map;
extern uint32_t mx_mapper_level_mask;
extern uint32_t mx_mapper_verbose;
extern int mx_mapper_msgbuf_size;
extern int mx_mapper_mapbuf_size;
extern uint32_t mx_small_message_threshold;
extern uint32_t mx_medium_message_threshold;
extern uint32_t mx_security_disabled;
extern int mx_kernel_mapper;
extern uint32_t mx_z_loopback;

#define MX_MAX_INSTANCE_DEFAULT 4

extern uint32_t mx_max_nodes;
extern uint32_t mx_max_endpoints;
extern uint32_t mx_min_light_endpoints;
extern uint32_t mx_max_light_endpoints;

#define MX_MAX_ENDPOINTS_DEFAULT 16

extern uint32_t mx_max_send_handles;
extern uint32_t mx_max_pull_handles;
extern uint32_t mx_max_push_handles;
extern uint32_t mx_max_rdma_windows;
extern uint32_t mx_cacheline_size;
extern uint32_t mx_intr_coal_delay;
extern uint32_t mx_override_e_to_f;
extern mx_spinlock_t mx_lanai_print_spinlock;
extern uint32_t mx_simulate_parity_error;
extern uint32_t mx_parity_recovery;
extern uint32_t mx_recover_from_all_errors;
extern uint32_t mx_pcie_down;
extern uint32_t mx_pcie_down_on_error;
extern uint32_t mx_max_host_queries;

/* sleeping and waking up */

#define MX_SLEEP_INTR    1
#define MX_SLEEP_NOINTR  2

/* for the watchdog to properly detect parity error (rather than such
   error ending up as some MX_DEAD_xxx_TIMEOUT), the final timeout
   should be:
   final-timeout >= (min-desired-timeout + 2*Watchdog)
   in addition to sram parity error, this avoid confusing root cause
   and secondary effects for any HW check done in the watchdog_body
*/

#define MX_SMALL_WAIT    100
#define MX_CLOSE_WAIT	 1000*15 /* 10 seconds */
#define MX_LOGGING_WAIT  MX_MAX_WAIT

#if MX_OS_UDRV
#define MX_MCP_INIT_TIMEOUT  (mx_lxgdb ? 60*1000 : 12*1000)
#define MX_COMMAND_WAIT      (mx_lxgdb ? 600*1000 : 12*1000)
#define MX_WATCHDOG_TIMEOUT  (mx_lxgdb ? 30 : 5)
#else
/* FIXME: do we need 10s timeout ? */
#define MX_MCP_INIT_TIMEOUT  12*1000
#define MX_COMMAND_WAIT      12*1000
#define MX_WATCHDOG_TIMEOUT  5
#endif


void mx_wake(mx_sync_t *);
int mx_sleep(mx_sync_t *, int, int flags);
mx_waitq_t *mx_new_waitq(void);
void mx_free_waitq(mx_waitq_t *);

/* syncs.. need to move elsewhere */

void mx_sync_init (mx_sync_t *s, mx_instance_state_t *is, int unique, 
		   char *string);
void mx_sync_reset (mx_sync_t *s);
void mx_sync_destroy (mx_sync_t *s);

#ifndef mx_spin_lock_destroy
void mx_spin_lock_destroy(mx_spinlock_t *s);
#endif

void mx_spin_lock_init(mx_spinlock_t *s, mx_instance_state_t *is, int unique, 
		       char *string);

#ifndef mx_mutex_enter
void mx_mutex_enter (mx_sync_t *s);
void mx_mutex_exit (mx_sync_t *s);
#endif

int mx_driver_wait(mx_endpt_state_t *, uint32_t);

extern mx_sync_t mx_global_mutex;

/* open/close/ioctl .. need to move elsewhere */


int mx_common_ioctl (mx_endpt_state_t *es, uint32_t cmd,
		     const mx_uaddr_t in);

int mx_endptless_ioctl (uint32_t cmd, const mx_uaddr_t in,
			uint32_t privileged, uint32_t is_kernel);

int mx_common_open (int32_t unit, int32_t endpoint, mx_endpt_state_t *es,
		    int raw);
void mx_common_close (mx_endpt_state_t *es);
int mx_open_mcp_endpoint(mx_endpt_state_t *es);

void mx_unpin_page(mx_instance_state_t *, mx_page_pin_t *, int);
void mx_unpin_vpages(mx_instance_state_t *, mx_page_pin_t *, int, int);
int mx_pin_page(mx_instance_state_t *, mx_page_pin_t *, int, uint64_t);
int mx_pin_vpages(mx_instance_state_t *, mx_page_pin_t *, mcp_dma_addr_t *, int, int, uint64_t);
mx_page_pin_t *mx_find_pin(mx_endpt_state_t *es,  
			   int (*testfp)(mx_page_pin_t *pin, void *arg), void *arg);

/* PCI accessor functions */
int mx_read_pci_config_8 (mx_instance_state_t *is,
			  uint32_t offset,
			  uint8_t *val);

int mx_read_pci_config_16 (mx_instance_state_t *is,
			   uint32_t offset,
			   uint16_t *val);

int mx_read_pci_config_32 (mx_instance_state_t *is,
			   uint32_t offset,
			   uint32_t *val);

int mx_write_pci_config_8 (mx_instance_state_t *is,
			   uint32_t offset,
			   uint8_t val);

int mx_write_pci_config_16 (mx_instance_state_t *is,
			    uint32_t offset,
			    uint16_t val);

int mx_write_pci_config_32 (mx_instance_state_t *is,
			    uint32_t offset,
			    uint32_t val);


#if MX_OS_UDRV
void mx_write_lanai_isr(struct mx_instance_state *is, uint32_t val);
void mx_write_lanai_ctrl_bit(struct mx_instance_state *is, int val);
#else
#define mx_write_lanai_isr(is,val) mx_write_lanai_special_reg_u32(is,lx.ISR, val)
#define mx_write_lanai_ctrl_bit(is,bit) mx_write_pci_config_32(is, 0x40, 1U << (bit))
/* or equivalent: is->lanai.control_regs[0x10] = htonl(1 << MX_LX_CTRL(bit)))) */
#endif

#define MX_MEM_HOSTMEM 0
#define MX_MEM_SRAM 1
#define MX_MEM_CONTROL 2
#define MX_MEM_SPECIAL 3

int mx_mmap_off_to_kva(mx_endpt_state_t *, unsigned long, void **, int *,
		       mx_page_pin_t **);


/* notify firmware command is pending by setting the REQACK1 bit */

void mx_lanai_command_push(mx_instance_state_t *is);
int mx_lanai_command(mx_instance_state_t *, uint32_t, uint32_t, uint32_t, 
		     uint32_t, uint32_t *, mx_sync_t *);
int mx_async_lanai_command(mx_instance_state_t *is, uint32_t cmd, uint32_t cmd_index, 
			   uint32_t cmd_data0, uint32_t cmd_data1);
int mx_wait_command(mx_instance_state_t *is, struct cmdq_entry *entry);

int mx_status_command(mx_instance_state_t *is, uint32_t mcp_status, 
		      uint32_t mcp_data);
int mx_submit_command(mx_instance_state_t *is, struct cmdq_entry *entry);
void mx_transfer_commands(mx_instance_state_t *is);
void mx_lanai_command_complete(mx_instance_state_t *is, uint32_t slot, uint32_t mcp_data, 
			       int mcp_status);
int mx_common_interrupt(mx_instance_state_t *is);


/* memory registration related functionality */
void mx_free_dma_win (mx_instance_state_t *is, mx_host_dma_win_t *hc);
mx_host_dma_win_t *mx_new_dma_win(mx_instance_state_t *is, int nsegs);

static inline int
mx_is_dead(mx_instance_state_t *is)
{
  return (is->flags & (MX_IS_DEAD | MX_PARITY_RECOVERY));
}
#define mx_digit(c) (((c) >= '0' && (c) <= '9') ? ((c) - '0') : \
                 (((c) >= 'A' && (c) <= 'F') ? (10 + (c) - 'A') : \
                  (((c) >= 'a' && (c) <= 'f') ? (10 + (c) - 'a') : -1)))

int mx_alloc_dma_page(mx_instance_state_t *is, char **alloc_addr, char **addr, mx_page_pin_t *pin);

void mx_free_dma_page(mx_instance_state_t *is, char **alloc_addr, mx_page_pin_t *pin);
int
mx_alloc_zeroed_dma_page(mx_instance_state_t *is, char **alloc_addr, 
			 char **addr, mx_page_pin_t *pin);

int mx_rand(void);
void mx_watchdog_body(void);

int mx_common_alloc_copyblock(mx_instance_state_t *is, mx_copyblock_t *cb);
void mx_common_free_copyblock(mx_instance_state_t *is, mx_copyblock_t *cb);
void mx_parse_mcp_error(mx_instance_state_t *is);


/* copyin and out */
static inline int
mx_copyin (mx_uaddr_t what, void *where, size_t amount, uint32_t is_kernel)
{
  if (is_kernel) {
    bcopy((void *)(uintptr_t)what, where, amount);
    return 0;
  } else
    return mx_arch_copyin(what, where, amount);
}

static inline int
mx_copyout (void *what, mx_uaddr_t where, size_t amount, uint32_t is_kernel)
{
  if (is_kernel) {
    bcopy(what, (void *)(uintptr_t)where, amount);
    return 0;
  } else
    return mx_arch_copyout(what, where, amount);
}

#if MX_DISABLE_ETHERNET
/* ethernet hooks into the rest of mx */
#define mx_ether_tx_done(is,index)
#define mx_ether_rx_done_small(is,count,len,csum,fl)
#define mx_ether_rx_done_big(is,count,len,csum,fl)
#define mx_ether_attach(is) 0
#define mx_ether_detach(is) 
#define mx_ether_parity_detach(is) 0
#define mx_ether_parity_reattach(is)
#define mx_ether_link_change_notify(is)
#else
void mx_ether_tx_done(mx_instance_state_t *is, uint32_t mcp_index);
void mx_ether_rx_done_small(mx_instance_state_t *is, int count, int len, int csum, int flags);
void mx_ether_rx_done_big(mx_instance_state_t *is, int count, int len, int csum, int flags);
int mx_ether_attach(mx_instance_state_t *is);
void mx_ether_detach(mx_instance_state_t *is);
int mx_ether_parity_detach(mx_instance_state_t *is);
void mx_ether_parity_reattach(mx_instance_state_t *is);
void mx_ether_link_change_notify(mx_instance_state_t *is);
#endif

int mx_instance_status_string(int unit, char **str, int *len);

int mx_start_mapper(mx_instance_state_t *is);
int mx_stop_mapper(mx_instance_state_t *is);
mx_endpt_state_t *mx_get_endpoint(mx_instance_state_t *is, int endpt);
void mx_put_endpoint(mx_endpt_state_t *es);
int mx_direct_get(mx_endpt_state_t *dst_es, mx_shm_seg_t *dst_segs, uint32_t dst_nsegs,
		  mx_endpt_state_t *src_es, mx_shm_seg_t *src_segs, uint32_t src_nsegs,
		  uint32_t length);
mx_instance_state_t *mx_get_instance(uint32_t unit);
void mx_release_instance(mx_instance_state_t *is);
void mx_mark_board_dead(mx_instance_state_t *is, int reason, int arg);
void mx_parse_eeprom_strings(mx_instance_state_t *is);

int mx_direct_get_common(mx_shm_seg_t *dst_segs, uint32_t dst_nsegs,
			 void * src_space, mx_shm_seg_t *src_segs, uint32_t src_nsegs,
			 uint32_t length);

/* OS specific callback for direct get, copying from another process
 * user-space to current process user-space.
 */
int mx_arch_copy_user_to_user(mx_uaddr_t udst,
			      mx_uaddr_t usrc, void * src_space,
			      uint32_t len);


int mx_pcie_link_reset(mx_instance_state_t *is);
uint16_t mx_bridge_pci_sec_status(mx_instance_state_t *is);
int mx_find_capability(mx_instance_state_t *is, unsigned cap_id);

#endif /* _mx_instance_h_ */
